using System;
using System.Collections.Generic;
using System.Data;
using System.Data.SqlTypes;
using System.Diagnostics.Eventing.Reader;
using System.Linq;
using System.Reflection;
using gov.va.med.vbecs.Common.Log;
using gov.va.med.vbecs.Common;
using gov.va.med.vbecs.DAL.HL7AL;
using gov.va.med.vbecs.DAL.HL7.OpenLibrary;
using gov.va.med.vbecs.DAL.HL7.OpenLibrary.Messages;

namespace gov.va.med.vbecs.DAL.HL7.Parsers
{

	#region Header

	//<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	//<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	//<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	//<Developers>
    //	<Developer>Russell Stephenson</Developer>
	//</Developers>
	//<SiteName>Hines OIFO</SiteName>
	//<CreationDate>1/18/2016</CreationDate>
	//<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	//<summary></summary>

	#endregion

	/// <summary>
	/// Auto Instrument HL7 message parser
	/// </summary>
	public class AutoInstrumentHL7Parser
	{
		private const string ORU_CLINICAL_ORDER_MESSAGE = "ORU^R01";
       

        private HL7ProtocolMessage _ackMessage;
        private HL7AIOruMessage _oruMessage;

        static BloodTestPanelName TestPanelNameToTestPanelEnum(string testPanelName)
        {
            Dictionary<string, int> testPanelNames = new Dictionary<string, int>();
            testPanelNames.Add("Unit ABORh",        (int)BloodTestPanelName.UnitABORh);
            testPanelNames.Add("Patient ABORh",     (int)BloodTestPanelName.PatientABORh);
            testPanelNames.Add("Antigen Typing",    (int)BloodTestPanelName.AnigenTyping);
            testPanelNames.Add("Weak D",            (int)BloodTestPanelName.WeakD);
            testPanelNames.Add("TAS",               (int)BloodTestPanelName.TAS);
            testPanelNames.Add("ABS",               (int)BloodTestPanelName.ABS);
            testPanelNames.Add("DAT",               (int)BloodTestPanelName.DAT);
            testPanelNames.Add("XM",                (int)BloodTestPanelName.XM);
            return (BloodTestPanelName)testPanelNames[testPanelName];
        }

		///<Developers>
		///	<Developer>Russell Stephenson</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/16/2016</CreationDate>
		///<TestCases>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
        /// BR_114.05	
        /// The message is processed by the VBECS system.  
        /// If the message is accepted (per the approved HL7 message profile specification), 
        /// a success reply is sent; otherwise, a rejection reply is sent to the broadcasting 
        /// application using the parameter tables in this UC_114. 
		/// </summary>
		/// <param name="intParms"><see cref="HL7Interface"/> containing interface parameters for the Auto Instrument interface.</param>
		/// <param name="messageString">HL7 message string.</param>
		/// <returns></returns>
		public HL7ProtocolMessage ParseHL7Message( HL7Interface intParms, string messageString )
		{
			if( intParms == null )
				throw( new ArgumentNullException( "intParms" ) );

			if( intParms.InterfaceName != InterfaceName.Auto_Instrument.ToString() )
                throw (new HL7Exception("AutoInstrumentHL7Parser.ParseHL7Message can only be used with the Auto_Instrument HL7Interface"));

			if( messageString == null || messageString == string.Empty )
				throw( new ArgumentNullException( "messageString" ) );

            
            _ackMessage = null;
            _oruMessage = null;


		    if (HL7Utility.ParseGetHL7MessageTypeAndTriggerEventString(messageString) == ORU_CLINICAL_ORDER_MESSAGE)
		    {
		        bool hL7StructureIsValid = false;
		        try
		        {
		            _oruMessage = new HL7AIOruMessage(messageString);
		            hL7StructureIsValid = true;

		            _ackMessage = ProcessOruMessage(intParms, _oruMessage);
                }
		        catch (Exception exc)
		        {
                    var messageBody = "Exception Message:\n" + HL7Utility.GetInnerExceptionFromException(exc);
                    //// EventLogAppender
                    //_eventsLogger.Error("Auto Instrument HL7 Parser: " + messageBody);
                    //// Duplicate error information in general logger
                    //_logger.Error("Auto Instrument HL7 Parser: " + messageBody);

		            _ackMessage = HL7AckMessage.BuildAckMessage(messageString, hL7StructureIsValid ? AckCodes.AE : AckCodes.AR,
		                messageBody);

		            InsertUpdateMessageLog(messageString, _ackMessage, MessageStatus.ProcessingError, intParms,
		                HL7Utility.GetInnerExceptionFromException(exc));

		            // CR 1928 - BNT 6/2/06 Removed HL7 Message text from email message.
                    //var mail = new HL7MailMessage();
                    //mail.SendMessage(intParms.InterfaceAdministratorEmail, PII                                    ",
                    //    "Auto_Instrument HL7 Parser caught System.Exception", messageBody, "smtp.DNS   ");

                    throw new Hl7AckException(_ackMessage, "Auto_Instrument HL7 Parser caught System.Exception", exc);
		        }
		    }
		    else
		    {
		        throw (new HL7Exception("Unsupported Message Type of (" +
		                                HL7Utility.ParseGetHL7MessageTypeAndTriggerEventString(messageString) +
		                                ") was received for the Auto Instrument Interface.\n\n" + "HL7 Message:\n" + messageString));
		    }
		    return _ackMessage;
		}

	    private static HL7ProtocolMessage ProcessOruMessage(HL7Interface intParms, HL7AIOruMessage message)
	    {
	        if (message == null)
	            throw (new ArgumentNullException("message"));


            HL7ProtocolMessage _ackMessage = null;
            HL7MessageLog.InsertMessageLog(HL7MessageLog.GetMessageDataForMessageLogInsert(message.GetMessage(),
                MessageStatus.ProcessingIncomingMessage, intParms, UpdateFunction.HL7AutoInstrumentInterface));

            var validationErrors = new List<string>();


	        if (intParms.InterfaceActiveIndicator)
	        {
	            DataRow divisionRow = Division.GetDivision(message.Msh.ReceivingFacility);
	            if (divisionRow != null && !(bool)divisionRow[VbecsTables.VamcDivision.ServiceTypeIndicator])
	            {
	                string errorString = string.Format("Divsion: {0} is configured as a transfusion only facility.",
	                    (string)divisionRow[VbecsTables.VamcDivision.DivisionName]);
                    validationErrors.Add(errorString);
	                errorString = validationErrors.Aggregate((workingSentence, next) => next + Environment.NewLine + workingSentence);
                    _ackMessage = HL7AckMessage.BuildAckMessage(message.Message, AckCodes.AE, errorString);
                    HL7MessageLog.InsertAckMessageControlId(message.MessageControlID, _ackMessage.GetMessageControlID(), MessageStatus.ProcessingError, intParms.InterfaceName, UpdateFunction.HL7AutoInstrumentInterface);
                    HL7MessageLog.UpdateMessageStatus(MessageStatus.ProcessingError, message.MessageControlID, errorString, 0, intParms.InterfaceName, UpdateFunction.HL7AutoInstrumentInterface);
                    return _ackMessage;
	            }
	        }
	        else
	        {
                string errorString = string.Format("Automated Interface is disabled.");
                validationErrors.Add(errorString);
                errorString = validationErrors.Aggregate((workingSentence, next) => next + Environment.NewLine + workingSentence);
                _ackMessage = HL7AckMessage.BuildAckMessage(message.Message, AckCodes.AE, errorString);
                HL7MessageLog.InsertAckMessageControlId(message.MessageControlID, _ackMessage.GetMessageControlID(), MessageStatus.ProcessingError, intParms.InterfaceName, UpdateFunction.HL7AutoInstrumentInterface);
                HL7MessageLog.UpdateMessageStatus(MessageStatus.ProcessingError, message.MessageControlID, errorString, 0, intParms.InterfaceName, UpdateFunction.HL7AutoInstrumentInterface);
                return _ackMessage;
	        }

	        BloodTestPanelName bloodTestPanelName = 0;
	        try
	        {
    	         bloodTestPanelName = TestPanelNameToTestPanelEnum(message.Obr.TestName);
	        }
            // *** Fortify Justified Code ***
            // *** We catch all exceptions by design. We do not suppress all exceptions silently. We record into the log every time we catch exception. So we know every time it happens and can trouble shoot accordingly. All exceptions should be caught to prevent service from failure.  ***
            catch (Exception e)
            {
                validationErrors.Add(string.Format("Invalid Test Name: [{0}] Exception: {1}.", message.Obr.TestName, e.Message));
            }
            // *** Fortify Justified Code *** 


            validationErrors = PendingResultsController.ProcessTestResults(message, bloodTestPanelName);

            if (validationErrors.Count > 0)
	        {
                var errorString = validationErrors.Aggregate((workingSentence, next) => next + Environment.NewLine + workingSentence);
                _ackMessage = HL7AckMessage.BuildAckMessage(message.Message, AckCodes.AE, errorString);
                HL7MessageLog.InsertAckMessageControlId(message.MessageControlID, _ackMessage.GetMessageControlID(), MessageStatus.ProcessingError, intParms.InterfaceName, UpdateFunction.HL7AutoInstrumentInterface);
                HL7MessageLog.UpdateMessageStatus(MessageStatus.ProcessingError, message.MessageControlID, errorString, 0, intParms.InterfaceName, UpdateFunction.HL7AutoInstrumentInterface);
            }
	        else
	        {
	            _ackMessage = HL7AckMessage.BuildAckMessage(message.Message, AckCodes.AA, string.Empty);
                HL7MessageLog.InsertAckMessageControlId(message.MessageControlID, _ackMessage.GetMessageControlID(), MessageStatus.SuccessfullyCompleted, intParms.InterfaceName, UpdateFunction.HL7AutoInstrumentInterface);
                HL7MessageLog.UpdateMessageStatus(MessageStatus.SuccessfullyCompleted, message.MessageControlID, string.Empty, 0, intParms.InterfaceName, UpdateFunction.HL7AutoInstrumentInterface);
            }

	        return _ackMessage;
	    }

	    private static void InsertUpdateMessageLog( string incomingMessage, HL7ProtocolMessage ackMessage, MessageStatus status, HL7Interface intParms, string errorText )
		{

            HL7MessageLog.InsertMessageLog(HL7MessageLog.GetMessageDataForMessageLogInsert(incomingMessage, status, intParms, UpdateFunction.HL7AutoInstrumentInterface));
			if(ackMessage != null)
			{

                HL7MessageLog.InsertAckMessageControlId(HL7Utility.GetMessageControlID(incomingMessage), ackMessage.GetMessageControlID(), status, intParms.InterfaceName, UpdateFunction.HL7AutoInstrumentInterface);
				
				// CR 2961
                HL7MessageLog.UpdateMessageStatus(status, HL7Utility.GetMessageControlID(incomingMessage), errorText, 0, intParms.InterfaceName, UpdateFunction.HL7AutoInstrumentInterface);
			}
		}

	}
}
